#!/bin/bash

#
# EECluster: An Energy-Efficient software tool for managing HPC Clusters
# (c) Universidad de Oviedo 2016
# 
# Autores: 
# 
# Alberto Cocaña Fernández
# Raquel Cortina Parajón
# José Ranilla Pastor
# Luciano Sánchez Ramos
# 
#

function ask_port {
   re_ask_port=true
	while $re_ask_port; do
		echo -n "Select a port to deploy the application [80]: "
		read port
		if [ ${#port} -le 0 ]; then
			port="80"
		fi

		test_port=$(netstat -tulpn | grep ":$port " 2> /dev/null)
		if [ ${#test_port} -gt 0 ]; then
			re_ask_port=true
			echo -e "\e[0;33m[WARNING]\e[0m Port $port is being used. Select other port number."
		else
			re_ask_port=false
		fi
	done
}


#========================================================================================
# Check EECluster requirements																			|
#========================================================================================


echo -e "---------------------------------------------------"
echo -e "EECluster requirements check"
echo -e "---------------------------------------------------"

#Check if ok or not
ok_check=true

#Check if script is running as root 
#(at the moment, only works with sudo su),
#Needs sudo for copying folders, but if it is executed as "sudo ./instalacion.sh", $SGE_ROOT is not found
if [[ $EUID -ne 0 ]]; then
   echo -e "\e[0;31m[ERROR] \e[0m Configure script must be run with root user" 
	exit
fi

#Check if javac is installed (full JDK, not just JRE)
var_javac=$(which javac 2> /dev/null)
if [ ${#var_javac} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m Java Development Kit 1.6 or higher is required but it could not be found"
	ok_check=false	
	#exit
else
	#Check if javac is 1.6 or higher (for glassfish)
	JAVAC_VER=$(javac -version 2>&1 | sed 's/javac \(.*\)\.\(.*\)\..*/\1\2/; 1q')
	
	if [ "$JAVAC_VER" -ge 17 ]; then
		echo -e "\e[0;32m[OK]\e[0m \t Java Development Kit 1.7 or higher was found"
	elif [ "$JAVAC_VER" -ge 16 ]; then
		echo -e "\e[0;33m[WARNING]\e[0m Java JDK 1.6 was found but Glassfish 4 needs Java Development Kit 1.7 or higher"
	else
		echo -e "\e[0;31m[ERROR] \e[0m Java Development Kit 1.6 or higher version is required to run EECluster"
	fi
fi

#Check if java is installed too
var_java=$(which java 2> /dev/null)
if [ ${#var_java} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m Java JRE 1.6 or higher is required but it could not be found"
	ok_check=false	
	#exit
else
	#Check if java is 1.6 or higher
   JAVA_VER=$(java -version 2>&1 | sed 's/java version "\(.*\)\.\(.*\)\..*"/\1\2/; 1q')
	
	if [ "$JAVA_VER" -ge 17 ]; then
		echo -e "\e[0;32m[OK]\e[0m \t Java JRE 1.7 or higher was found"
	elif [ "$JAVA_VER" -ge 16 ]; then
		echo -e "\e[0;33m[WARNING]\e[0m Java JRE 1.6 was found but Glassfish 4 needs Java JRE 1.7 or higher"
	else
		echo -e "\e[0;31m[ERROR] \e[0m Java JRE 1.6 or higher version is required to run EECluster"
	fi
fi


#Check if SGE is installed (and used components qhost, qacct, qconf and qstat)
var_SGE=$(env | grep "SGE_ROOT")
#var_SGE=$(sudo su root -c echo $SGE_ROOT)
#echo -n "SGE: "
#echo $var_SGE
# echo $var_SGE
if [ ${#var_SGE} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m SGE is required but it could not be found"
	ok_check=false	
	#exit
else
	var_qhost=$(which qhost 2> /dev/null)
	var_qacct=$(which qacct 2> /dev/null)
	var_qconf=$(which qconf 2> /dev/null)
	var_qstat=$(which qstat 2> /dev/null)
	if [ ${#var_qhost} -le 0 ]; then
		echo -e "\e[0;31m[ERROR] \e[0m Cannot find qhost"
		ok_check=false
		#exit
	elif [ ${#var_qacct} -le 0 ]; then
		echo -e "\e[0;31m[ERROR] \e[0m Cannot find qacct"
		ok_check=false
		#exit
	elif [ ${#var_qconf} -le 0 ]; then
		echo -e "\e[0;31m[ERROR] \e[0m Cannot find qconf"
		ok_check=false
		#exit
	elif [ ${#var_qstat} -le 0 ]; then
		echo -e "\e[0;31m[ERROR] \e[0m Cannot find qstat"
		ok_check=false
		#exit
	else
		echo -e "\e[0;32m[OK]\e[0m \t SGE is correctly installed"
	fi
fi

#Check if etherwake is installed
#var_etherwake=$(dpkg --get-selections | grep "etherwake")
var_etherwake=$(which etherwake 2> /dev/null)
var_ether_wake=$(which ether-wake 2> /dev/null)
if [ ${#var_etherwake} -le 0 ] && [ ${#var_ether_wake} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m etherwake is required but it could not be found"
	ok_check=false
	#exit
else
	echo -e "\e[0;32m[OK]\e[0m \t etherwake is correctly installed"
fi

#Check if ethtool is installed
#var_ethtool=$(dpkg --get-selections | grep "ethtool")
var_ethtool=$(which ethtool 2> /dev/null)
if [ ${#var_ethtool} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m ethtool is required but it could not be found"
	ok_check=false
	#exit
else
	echo -e "\e[0;32m[OK]\e[0m \t ethtool is correctly installed"
fi

#Check if ipmiutil is installed
var_ipmiutil=$(which ipmiutil 2> /dev/null)
if [ ${#var_ipmiutil} -le 0 ]; then
	echo -e "\e[0;33m[WARNING]\e[0m ipmiutil is not found"
	#ok_check=false
	#exit
else
	echo -e "\e[0;32m[OK]\e[0m \t ipmiutil is correctly installed"
fi

#Check if ssh is installed (client and server?)
#Client
var_ssh=$(which ssh 2> /dev/null)
if [ ${#var_ssh} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m ssh client is required but it could not be found"
	ok_check=false
	#exit
else
	echo -e "\e[0;32m[OK]\e[0m \t ssh client is correctly installed"
fi


#Check if MySQL is installed (client and server?)
#Client
var_mysql=$(which mysql 2> /dev/null)
if [ ${#var_mysql} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m MySQL command-line tool is required but it could not be found"
	ok_check=false
	#exit
else
	echo -e "\e[0;32m[OK]\e[0m \t MySQL command-line tool is correctly installed"
fi

#Server
var_mesqld=$(which mysqld 2> /dev/null)
if [ ${#var_mesqld} -le 0 ]; then
	#echo -e "\e[0;31m[ERROR] \e[0m MySQL service is required but it could not be found"
	which_mysqld=false
	#exit
else
	echo -e "\e[0;32m[OK]\e[0m \t MySQL service is correctly installed"
	which_mysqld=true
fi

#Check if MySQL server is running
#Check if MySQL server is running
var_service=$(which service 2> /dev/null)
var_mysql_service=""

if [ ${#var_service} -le 0 ]; then
	if [ -e /etc/init.d/mysql ]; then
		var_mysql_service=$(/etc/init.d/mysql status)
		#echo "/etc/init.d/mysql"
	elif [ -e /etc/init.d/mysqld ]; then
		var_mysql_service=$(/etc/init.d/mysqld status)
		#echo "/etc/init.d/mysqld"
	fi
else
	var_service_mysql=$(service mysql status 2> /dev/null)
	var_service_mysqld=$(service mysqld status 2> /dev/null)
	if [[ $var_service_mysql != "" ]]; then
		var_mysql_service=$(service mysql status)
		#echo "service mysql"
	elif [[ $var_service_mysqld != "" ]]; then
		var_mysql_service=$(service mysqld status)
		#echo "service mysqld"
	else
		if [ $which_mysqld == false ]; then
			echo -e "\e[0;31m[ERROR] \e[0m MySQL service is required but it could not be found"
			ok_check=false
		else	
			echo -e "\e[0;33m[WARNING]\e[0m Cannot confirm if MySQL service is already running"
		fi
	fi		
fi

if [[ ${var_mysql_service} == *"pid"* ]] || [[ ${var_mysql_service} == *"running"* ]]; then
	echo -e "\e[0;32m[OK]\e[0m \t MySQL is running"
else
	echo -e "\e[0;31m[ERROR] \e[0m MySQL is not running. Cannot create EECluster database"
   ok_check=false
	#exit
fi


#Check if cat is installed
var_cat=$(which cat 2> /dev/null)
if [ ${#var_cat} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m cat is required but it could not be found"
	ok_check=false
	#exit
else
	echo -e "\e[0;32m[OK]\e[0m \t cat is correctly installed"
fi

#Check if arp is installed
var_arp=$(which arp 2> /dev/null)
if [ ${#var_arp} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m arp is required but it could not be found"
	ok_check=false
	#exit
else
	echo -e "\e[0;32m[OK]\e[0m \t arp is correctly installed"
fi

#Check if ping is installed
var_ping=$(which ping 2> /dev/null)
if [ ${#var_ping} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m ping is required but it could not be found"
	ok_check=false
	#exit
else
	echo -e "\e[0;32m[OK]\e[0m \t ping is correctly installed"
fi

#Checks if all checkings are ok
if(! $ok_check); then
	exit
fi

#========================================================================================
# Deploy EECluster																		|
#========================================================================================

echo -e "\n"
echo -e "---------------------------------------------------"
echo -e "EECluster deployment"
echo -e "---------------------------------------------------"


echo -e "\n"
echo -n "Enter EEClusterd install path [/usr/share]: "
read eecluster_path
if [ ${#eecluster_path} -le 0 ]; then
	eecluster_path="/usr/share"
fi

if [ ! -d "$eecluster_path" ]; then
	echo -e "\e[0;31m[ERROR] \e[0m Directory $eecluster_path doesn't exist, introduce a correct existent path for EECluster install"
	exit
fi

#Create EECLUSTER_PATH
if grep -q "EECLUSTER_PATH" "/etc/environment"; then
	sed -i "s|EECLUSTER_PATH=.*|EECLUSTER_PATH=${eecluster_path}/eeclusterd|g" /etc/environment
else
	echo "EECLUSTER_PATH=$eecluster_path/eeclusterd" >> /etc/environment
fi


#Copy service folder and daemon
 cp -a -r eeclusterd ${eecluster_path}/eeclusterd
 cp -a init.d/eeclusterd /etc/init.d/eeclusterd
 sed -i "s|FILE_PATH=.*|FILE_PATH=\"${eecluster_path}/eeclusterd\"|g" /etc/init.d/eeclusterd
 sed -i "s|JAVA_HOME=.*|JAVA_HOME=\"${var_java}\"|g" /etc/init.d/eeclusterd
 cp -a HGFS_tuner.sh ${eecluster_path}/eeclusterd/HGFS_tuner.sh
 cp -a tuning.properties ${eecluster_path}/eeclusterd/tuning.properties
 chmod +s ${eecluster_path}/eeclusterd/HGFS_tuner.sh

#========================================================================================
# Configure EECluster																	|
#========================================================================================

echo -e "\n"
echo -e "---------------------------------------------------"
echo -e "EECluster configuration"
echo -e "---------------------------------------------------"

#------------------------------------------------------------------------
# Create EECluster database and tables							
#------------------------------------------------------------------------

echo -n 'Enter MySQL admin user: '
read mysql_user
echo -n 'Enter MySQL admin password: '
read -s mysql_password
echo ""

# Test connection
mysql_conn_test=$(mysql -h localhost -u $mysql_user -p$mysql_password -e "select 1" 2> /dev/null)
if [ ${#mysql_conn_test} -le 0 ]; then
	echo -e "\e[0;31m[ERROR] \e[0m Access denied for user '${mysql_user}'"
	exit
fi

# Create eecluster user
password_error=true
while $password_error; do
	echo -n 'Enter MySQL eecluster user: '
	read mysql_eecluster_user
	echo -n 'Enter MySQL eecluster user password: '
	read -s mysql_eecluster_password
	echo ""
	echo -n 'Confirm MySQL eecluster user password: '
	read -s mysql_eecluster_password_confirm
	echo ""

	if [[ ${mysql_eecluster_password} == ${mysql_eecluster_password_confirm} ]]; then
		password_error=false
	else
		echo -e "\e[0;31m[ERROR] \e[0m Passwords don't match."
		password_error=true
	fi
done

if [[ ${mysql_eecluster_user} == "" ]] || [[ ${mysql_eecluster_password} == "" ]]; then
	echo -e "\e[0;31m[ERROR] \e[0m Username and password can't be empty"
	exit
fi

mysql_eecluster_user_test=$(mysql -h localhost -u ${mysql_user} -p${mysql_password} -e "USE mysql; SELECT user FROM user WHERE user='${mysql_eecluster_user}'" 2> /dev/null)

if [ ${#mysql_eecluster_user_test} -gt 0 ]; then
	re_ask_mysql_eecluster_user=true
else
	re_ask_mysql_eecluster_user=false
fi

while $re_ask_mysql_eecluster_user; do
	echo -e "\e[0;33m[WARNING]\e[0m User '$mysql_eecluster_user' already exists in mysql database."

	mysql_eecluster_password_test=$(mysql -h localhost -u ${mysql_user} -p${mysql_password} -e "SELECT user FROM mysql.user WHERE user='$mysql_eecluster_user' and password=PASSWORD('$mysql_eecluster_password')" 2> /dev/null)

	if [ ${#mysql_eecluster_password_test} -gt 0 ]; then
		echo -n -e "\033[1m[INFO] \e[0m Password is correct. Do you want to give privileges to user '$mysql_eecluster_user'? (y/n) [y]: "
		read givePrivileges

		if [ ${#givePrivileges} -le 0 ]; then
			givePrivileges="y"
		fi

		if [[ ${givePrivileges} == "y" ]] || [[ ${givePrivileges} == "yes" ]]; then		
			re_ask_mysql_eecluster_user=false
		else
			echo -e "\033[1m[INFO] \e[0m Must introduce new user."
			re_ask_mysql_eecluster_user=true
		fi
	else
		echo -e "\e[0;31m[ERROR] \e[0m Password is incorrect for user '$mysql_eecluster_user'. Please introduce new user."
		re_ask_mysql_eecluster_user=true
	fi

	if($re_ask_mysql_eecluster_user); then

		password_error=true
		while $password_error; do
			echo -n 'Enter MySQL eecluster user: '
			read mysql_eecluster_user
			echo -n 'Enter MySQL eecluster user password: '
			read -s mysql_eecluster_password
			echo ""
			echo -n 'Confirm MySQL eecluster user password: '
			read -s mysql_eecluster_password_confirm
			echo ""

			if [[ ${mysql_eecluster_password} == ${mysql_eecluster_password_confirm} ]]; then
				password_error=false
			else
				echo -e "\e[0;31m[ERROR] \e[0m Passwords don't match."
				password_error=true
			fi
		done

		if [[ ${mysql_eecluster_user} == "" ]] || [[ ${mysql_eecluster_password} == "" ]]; then
			echo -e "\e[0;31m[ERROR] \e[0m Username and password can't be empty"
			exit
		fi

		mysql_eecluster_user_test=$(mysql -h localhost -u ${mysql_user} -p${mysql_password} -e "USE mysql; SELECT user FROM user WHERE user='${mysql_eecluster_user}'" 2> /dev/null)

		if [ ${#mysql_eecluster_user_test} -gt 0 ]; then
			re_ask_mysql_eecluster_user=true
		else
			re_ask_mysql_eecluster_user=false
		fi
	fi
done

#Create eecluster mysql user
mysql -u $mysql_user -p$mysql_password -e "CREATE USER '${mysql_eecluster_user}'@'localhost' IDENTIFIED BY '${mysql_eecluster_password}'" 2> /dev/null


# Grant all privileges to eecluster user
mysql -u $mysql_user -p$mysql_password -e "GRANT ALL ON mysql.user to '${mysql_eecluster_user}'@'localhost';" 2> /dev/null
mysql -u $mysql_user -p$mysql_password -e "GRANT ALL ON db_eecluster.* to '${mysql_eecluster_user}'@'localhost';" 2> /dev/null
mysql -u $mysql_user -p$mysql_password -e "GRANT ALL ON db_eecluster_jdbcrealm.* to '${mysql_eecluster_user}'@'localhost';" 2> /dev/null
mysql -u $mysql_user -p$mysql_password -e "FLUSH PRIVILEGES" 2> /dev/null

# Create tables
old_tables=$(mysql -h localhost -u ${mysql_user} -p${mysql_password} -e "USE db_eecluster; SELECT * FROM host" 2> /dev/null)

if [ ${#old_tables} -gt 0 ]; then
	echo -n "Reset EECluster tables and info? (y/n) [y]: "
	read deleteTables
	if [ ${#deleteTables} -le 0 ]; then
		deleteTables="y"
	fi

	if [[ ${deleteTables} == "y" ]] || [[ ${deleteTables} == "yes" ]]; then
		mysql -u $mysql_user -p$mysql_password -e "source create_tables.sql" 2> /dev/null
	fi
else
	mysql -u $mysql_user -p$mysql_password -e "source create_tables.sql" 2> /dev/null
fi





# Default dashboard user (admin:admin)
echo -e "\033[1m[INFO]\e[0m Creating default dashboard user (admin:admin)"
 mysql -u $mysql_user -p$mysql_password -e "INSERT INTO db_eecluster_jdbcrealm.users (username, password, securityGroup) VALUES ('admin', '8c6976e5b5410415bde908bd4dee15dfb167a9c873fc4bb8a81f6f2ab448a918', 0);" 2> /dev/null

#------------------------------------------------------------------------
# Generate configuration.properties file								
#------------------------------------------------------------------------

# Control horizon measured in seconds
echo -n "Enter time interval between cluster reconfigurations (seconds) [15]: "
read controlHorizon
if [ ${#controlHorizon} -le 0 ]; then
	controlHorizon="15"
fi

	#Check if it is a positive integer
	if [[ ! $controlHorizon =~ ^[\-0-9]+$ ]] || (( controlHorizon <= 0)); then
		controlHorizon="15"
		echo -e "\e[0;33m[WARNING]\e[0m Must introduce a positive integer. Default value 15 has been set"
	fi

# Host start timeout (seconds)
echo -n "Enter host power-on timeout value (seconds) [200]: "
read startTimeout
if [ ${#startTimeout} -le 0 ]; then
	startTimeout="200"
fi

	#Check if it is a positive integer
	if [[ ! $startTimeout =~ ^[\-0-9]+$ ]] || (( startTimeout <= 0)); then
		startTimeout="200"
		echo -e "\e[0;33m[WARNING]\e[0m Must introduce a positive integer. Default value 200 has been set"
	fi

# Host start timeout (seconds)
echo -n "Enter host Shutdown timeout value (seconds) [320]: "
read shutdownTimeout
if [ ${#shutdownTimeout} -le 0 ]; then
	shutdownTimeout="320"
fi

	#Check if it is a positive integer
	if [[ ! $shutdownTimeout =~ ^[\-0-9]+$ ]] || (( shutdownTimeout <= 0)); then
		shutdownTimeout="320"
		echo -e "\e[0;33m[WARNING]\e[0m Must introduce a positive integer. Default value 320 has been set"
	fi

# Node max workload threshold to refresh power consumption
echo -n "Enter node max workload threshold to refresh power consumption [1.5]: "
read loadThresholdPowerRead
if [ ${#loadThresholdPowerRead} -le 0 ]; then
   loadThresholdPowerRead="1.5"
fi

	#Check if it is a positive number or zero 
			#¿it could be zero?
   if ( ! [[ $loadThresholdPowerRead =~ ^[0-9]+\.?[0-9]*$ ]] ); then 
		loadThresholdPowerRead="1.5"
		echo -e "\e[0;33m[WARNING]\e[0m Must introduce a positive number. Default value 1.5 has been set"
	fi

configuration_name="${eecluster_path}/eeclusterd/conf/configuration.properties"

# Erase file content
echo "" > ${configuration_name}

# Write parameters to file
echo "# Control horizon measured in seconds" 							>> ${configuration_name}
echo "controlHorizon = ${controlHorizon}" 								>> ${configuration_name}
echo ""																	>> ${configuration_name}
echo "# Host power-on timeout value (seconds)" 							>> ${configuration_name}
echo "startTimeout = ${startTimeout}"									>> ${configuration_name}
echo ""																	>> ${configuration_name}
echo "# host Shutdown timeout value (seconds)" 							>> ${configuration_name}
echo "shutdownTimeout = ${shutdownTimeout}"								>> ${configuration_name}
echo ""																	>> ${configuration_name}
echo "# node max workload threshold to refresh power consumption" 		>> ${configuration_name}
echo "loadThresholdPowerRead = ${loadThresholdPowerRead}"				>> ${configuration_name}
echo ""																	>> ${configuration_name}

#------------------------------------------------------------------------
# Generate glassfish-resources.xml file								
#------------------------------------------------------------------------

glassfish_resources_path="glassfish-resources.xml"

echo '<?xml version="1.0" encoding="UTF-8"?>' > ${glassfish_resources_path}
echo '<!DOCTYPE resources PUBLIC "-//GlassFish.org//DTD GlassFish Application Server 3.1 Resource Definitions//EN" "http://glassfish.org/dtds/glassfish-resources_1_5.dtd">' >> ${glassfish_resources_path}
echo '<resources>' >> ${glassfish_resources_path}
echo '    <jdbc-connection-pool allow-non-component-callers="false" associate-with-thread="false" connection-creation-retry-attempts="0" connection-creation-retry-interval-in-seconds="10" connection-leak-reclaim="false" connection-leak-timeout-in-seconds="0" connection-validation-method="auto-commit" datasource-classname="com.mysql.jdbc.jdbc2.optional.MysqlDataSource" fail-all-connections="false" idle-timeout-in-seconds="300" is-connection-validation-required="false" is-isolation-level-guaranteed="true" lazy-connection-association="false" lazy-connection-enlistment="false" match-connections="false" max-connection-usage-count="0" max-pool-size="32" max-wait-time-in-millis="60000" name="MySQLPool" non-transactional-connections="false" pool-resize-quantity="2" res-type="javax.sql.DataSource" statement-timeout-in-seconds="-1" steady-pool-size="8" validate-atmost-once-period-in-seconds="0" wrap-jdbc-objects="false">' >> ${glassfish_resources_path}
echo '         <property name="URL" value="jdbc:mysql://localhost:3306/db_eecluster"/>' >> ${glassfish_resources_path}
echo '         <property name="User" value="'${mysql_eecluster_user}'"/>' >> ${glassfish_resources_path}
echo '         <property name="Password" value="'${mysql_eecluster_password}'"/>' >> ${glassfish_resources_path}
echo '    </jdbc-connection-pool>' >> ${glassfish_resources_path}
echo '' >> ${glassfish_resources_path}
echo '    <jdbc-resource pool-name="MySQLPool" jndi-name="jdbc/MySQLDS"></jdbc-resource>' >> ${glassfish_resources_path}
echo '' >> ${glassfish_resources_path}
echo '    <jdbc-connection-pool allow-non-component-callers="false" associate-with-thread="false" connection-creation-retry-attempts="0" connection-creation-retry-interval-in-seconds="10" connection-leak-reclaim="false" connection-leak-timeout-in-seconds="0" connection-validation-method="auto-commit" datasource-classname="com.mysql.jdbc.jdbc2.optional.MysqlDataSource" fail-all-connections="false" idle-timeout-in-seconds="300" is-connection-validation-required="false" is-isolation-level-guaranteed="true" lazy-connection-association="false" lazy-connection-enlistment="false" match-connections="false" max-connection-usage-count="0" max-pool-size="32" max-wait-time-in-millis="60000" name="MySQLSecurityConnectionPool" non-transactional-connections="false" pool-resize-quantity="2" res-type="javax.sql.DataSource" statement-timeout-in-seconds="-1" steady-pool-size="8" validate-atmost-once-period-in-seconds="0" wrap-jdbc-objects="false">' >> ${glassfish_resources_path}
echo '         <property name="URL" value="jdbc:mysql://localhost:3306/db_eecluster_jdbcrealm"/>' >> ${glassfish_resources_path}
echo '         <property name="User" value="'${mysql_eecluster_user}'"/>' >> ${glassfish_resources_path}
echo '         <property name="Password" value="'${mysql_eecluster_password}'"/>' >> ${glassfish_resources_path}
echo '    </jdbc-connection-pool>' >> ${glassfish_resources_path}
echo '' >> ${glassfish_resources_path}
echo '    <jdbc-resource pool-name="MySQLSecurityConnectionPool" jndi-name="jdbc/MySQLSecurityDS"></jdbc-resource>' >> ${glassfish_resources_path}
echo '' >> ${glassfish_resources_path}
echo '</resources>' >> ${glassfish_resources_path}
echo '' >> ${glassfish_resources_path}


#------------------------------------------------------------------------
# Generate jdbc_config.properties file								
#------------------------------------------------------------------------

jdbc_path="${eecluster_path}/eeclusterd/conf/jdbc_config.properties"

echo "" > ${jdbc_path}
echo "USER = ${mysql_eecluster_user}" 									>> ${jdbc_path}
echo "PASSWORD = ${mysql_eecluster_password}" 							>> ${jdbc_path}

echo "" 																>> ${jdbc_path}
echo "URL = jdbc:mysql://localhost:3306/db_eecluster?autoReconnect=true" >> ${jdbc_path}
echo "" 																>> ${jdbc_path}
echo "JDBC_DRIVER = com.mysql.jdbc.Driver" 								>> ${jdbc_path}
echo "" 																>> ${jdbc_path}
echo "# Error codes" 													>> ${jdbc_path}
echo "DUPLICATED_KEY = 1062" 											>> ${jdbc_path}
echo "" 																>> ${jdbc_path}


#------------------------------------------------------------------------
# Generate paths.properties file								
#------------------------------------------------------------------------

paths_path="${eecluster_path}/eeclusterd/conf/paths.properties"

echo ""								 								> ${paths_path}
echo "$(env | grep SGE_ROOT)"									>> ${paths_path}
echo "QHOST = $(which qhost 2> /dev/null)" 				>> ${paths_path}
echo "QACCT = $(which qacct 2> /dev/null)" 				>> ${paths_path}
echo "QCONF = $(which qconf 2> /dev/null)" 				>> ${paths_path}
echo "QSTAT = $(which qstat 2> /dev/null)" 				>> ${paths_path}
var_etherwake=$(which etherwake 2> /dev/null)
if [ ${#var_etherwake} -le 0 ]; then
	var_etherwake=$(which ether-wake 2> /dev/null)
fi
echo ""																>> ${paths_path}
echo "ETHERWAKE = $var_etherwake" 							>> ${paths_path}
echo ""																>> ${paths_path}
echo "ETHTOOL = $(which ethtool 2> /dev/null)"			>> ${paths_path}
var_ipmiutil=$(which ipmiutil 2> /dev/null)
if [ ${#var_ipmiutil} -gt 0 ]; then
	echo ""															>> ${paths_path}
	echo "IPMIUTIL = $var_ipmiutil"							>> ${paths_path}
fi
echo ""																>> ${paths_path}
echo "SSH = $(which ssh 2> /dev/null)"						>> ${paths_path}
echo ""																>> ${paths_path}
echo "CAT = $(which cat 2> /dev/null)"						>> ${paths_path}
echo ""																>> ${paths_path}
echo "ARP = $(which arp 2> /dev/null)"						>> ${paths_path}
echo ""																>> ${paths_path}
echo "PING = $(which ping 2> /dev/null)"					>> ${paths_path}
echo ""																>> ${paths_path}
echo "SLEEP = $(which sleep 2> /dev/null)"				>> ${paths_path}
echo ""	

#========================================================================================
# Start EECluster																	|
#========================================================================================

echo -n "Deploy EECluster web application in glassfish default domain? (y/n) [y]: "
read deployApp
if [ ${#deployApp} -le 0 ]; then
	deployApp="y"
fi
if [[ ${deployApp} == "y" ]] || [[ ${deployApp} == "yes" ]]; then

	echo -e ""
	echo -n "Enter Glassfish path: "
	read glassfish_path
	if [ ${#glassfish_path} -le 0 ]; then
		echo -e "\e[0;31m[ERROR] \e[0m Must specify a path"
		exit
	else
		if [ ! -e ${glassfish_path}/bin/asadmin ]
		then
			echo -e "\e[0;31m[ERROR] \e[0m Must specify a correct path where glassfish is installed"
			exit
		else
			echo -e "\e[0;32m[OK]\e[0m \t Glassfish path is correct"
		fi
	fi

   chmod +x ${glassfish_path}/bin/asadmin
   
   var_exists_domainEECluster=$(${glassfish_path}/bin/asadmin list-domains | grep domainEECluster 2> /dev/null)
   if [ ${#var_exists_domainEECluster} -gt 0 ]; then

		echo -n "The domain 'domainEECluster' already exists. Do you want to delete it? (y/n) [y]: "
		read deleteDomain
		if [ ${#deleteDomain} -le 0 ]; then
			deleteDomain="y"
		fi
		if [[ ${deleteDomain} == "y" ]] || [[ ${deleteDomain} == "yes" ]]; then
			echo -e "\033[1m[INFO] \e[0m Deleting old domainEECluster"
			
			${glassfish_path}/bin/asadmin undeploy EECluster

			var_running_domainEECluster=$(${glassfish_path}/bin/asadmin list-domains | grep "domainEECluster running" 2> /dev/null)			
			if [ ${#var_running_domainEECluster} -gt 0 ]; then
		      ${glassfish_path}/bin/asadmin stop-domain domainEECluster
		   fi
			
			${glassfish_path}/bin/asadmin delete-domain domainEECluster

			ask_port

			echo -e "\033[1m[INFO] \e[0m Creating domain domainEECluster"
			${glassfish_path}/bin/asadmin create-domain --instanceport $port domainEECluster

			echo -e "\033[1m[INFO] \e[0m Starting domain domainEECluster"
			${glassfish_path}/bin/asadmin start-domain domainEECluster
		else
			echo -e "\e[0;33m[WARNING]\e[0m The deployment of EECluster without deleting the old domain could fail."
			var_running_domainEECluster=$(${glassfish_path}/bin/asadmin list-domains | grep "domainEECluster running" 2> /dev/null)
			${glassfish_path}/bin/asadmin undeploy EECluster
		fi      
	else
		ask_port
		
		echo -e "\033[1m[INFO] \e[0m Creating domain domainEECluster"
		${glassfish_path}/bin/asadmin create-domain --instanceport $port domainEECluster

		echo -e "\033[1m[INFO] \e[0m Starting domain domainEECluster"
		${glassfish_path}/bin/asadmin start-domain domainEECluster
	fi

   
   echo -e "\033[1m[INFO] \e[0m Copy mysql-connector"
	cp -f mysql-connector-java-5.1.23-bin.jar ${glassfish_path}/glassfish/domains/domainEECluster/lib
   cp -f mysql-connector-java-5.1.23-bin.jar ${glassfish_path}/glassfish/lib
   echo -e "\033[1m[INFO] \e[0m Add-resources"	
   mv -f glassfish-resources.xml ${glassfish_path}/glassfish/domains/domainEECluster/glassfish-resources.xml 
   ${glassfish_path}/bin/asadmin add-resources ${glassfish_path}/glassfish/domains/domainEECluster/glassfish-resources.xml 


   echo -e "\033[1m[INFO] \e[0m Deploy EECluster"
	cp EECluster.war ${glassfish_path}/glassfish/domains/domainEECluster/EECluster.war
	${glassfish_path}/bin/asadmin deploy --force ${glassfish_path}/glassfish/domains/domainEECluster/EECluster.war

	#------------------------------------------------------------------------
	# Generate glassfish daemon						
	#------------------------------------------------------------------------

		glassfish_daemon_path="/etc/init.d/glassfish_eecluster"

		echo "" > ${glassfish_daemon_path}
		echo "#!/bin/bash" 											>> ${glassfish_daemon_path}
		echo "# description: Glassfish Start Stop Restart"	>> ${glassfish_daemon_path}
		echo "# processname: glassfish"							>> ${glassfish_daemon_path}
		echo "# chkconfig: 2345 20 80"								>> ${glassfish_daemon_path}
		echo "#"															>> ${glassfish_daemon_path}
		echo "### BEGIN INIT INFO"									>> ${glassfish_daemon_path}
		echo "# Provides:          glassfish"					>> ${glassfish_daemon_path}
		echo "# Required-Start:    eeclusterd"					>> ${glassfish_daemon_path}
		echo "# Required-Stop:     eeclusterd"					>> ${glassfish_daemon_path}
		echo "# Short-Description: Starts glassfish service"	>> ${glassfish_daemon_path}
		echo "# Description:       This file is used to start the daemon"	>> ${glassfish_daemon_path}
		echo "#                    and should be placed in /etc/init.d"	>> ${glassfish_daemon_path}
		echo "### END INIT INFO"									>> ${glassfish_daemon_path}
		echo ""															>> ${glassfish_daemon_path}
		echo "GLASSFISH_HOME=${glassfish_path}"				>> ${glassfish_daemon_path}
		echo ""															>> ${glassfish_daemon_path}
		echo "case \$1 in"											>> ${glassfish_daemon_path}
		echo "start)"													>> ${glassfish_daemon_path}
		echo "sh \$GLASSFISH_HOME/bin/asadmin start-domain domainEECluster"	>> ${glassfish_daemon_path}
		echo ";;"														>> ${glassfish_daemon_path}
		echo "stop)"													>> ${glassfish_daemon_path}
		echo "sh \$GLASSFISH_HOME/bin/asadmin stop-domain domainEECluster"	>> ${glassfish_daemon_path}
		echo ";;"														>> ${glassfish_daemon_path}
		echo "restart)"												>> ${glassfish_daemon_path}
		echo "sh \$GLASSFISH_HOME/bin/asadmin stop-domain domainEECluster"	>> ${glassfish_daemon_path}
		echo "sh \$GLASSFISH_HOME/bin/asadmin start-domain domainEECluster"	>> ${glassfish_daemon_path}
		echo ";;"														>> ${glassfish_daemon_path}
		echo "esac"														>> ${glassfish_daemon_path}

	chmod +x /etc/init.d/eeclusterd
   chmod +x /etc/init.d/glassfish_eecluster

   var_chkconfig=$(which chkconfig 2> /dev/null)
   if [ ${#var_chkconfig} -le 0 ]; then
		update-rc.d eeclusterd defaults 2> /dev/null
	   update-rc.d glassfish_eecluster defaults 2> /dev/null
   else
		chkconfig --add eeclusterd
      chkconfig --level 2345 eeclusterd on
	   chkconfig --add glassfish_eecluster
      chkconfig --level 2345 glassfish_eecluster on
   fi
	
else
	echo -e "\033[1m[INFO]\e[0m Must deploy the application manually"
fi



