/* Copyright 2009-2013 David Hadka
 *
 * This file is part of the MOEA Framework.
 *
 * The MOEA Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * The MOEA Framework is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the MOEA Framework.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.moeaframework.analysis.sensitivity;

import java.io.File;
import java.io.PrintStream;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.moeaframework.core.Problem;
import org.moeaframework.core.spi.ProblemFactory;
import org.moeaframework.util.CommandLineUtility;

/**
 * Command line utility for counting the number of entries in a result file.
 */
public class ResultFileInfo extends CommandLineUtility {
	
	/**
	 * Constructs the command line utility for counting the number of entries
	 * in a result file.
	 */
	public ResultFileInfo() {
		super();
	}
	
	@SuppressWarnings("static-access")
	@Override
	public Options getOptions() {
		Options options = super.getOptions();
		
		OptionGroup group = new OptionGroup();
		group.setRequired(true);
		group.addOption(OptionBuilder
				.withLongOpt("problem")
				.hasArg()
				.withArgName("name")
				.create('b'));
		group.addOption(OptionBuilder
				.withLongOpt("dimension")
				.hasArg()
				.withArgName("number")
				.create('d'));
		options.addOptionGroup(group);
		
		options.addOption(OptionBuilder
				.withLongOpt("output")
				.hasArg()
				.withArgName("file")
				.create('o'));
		
		return options;
	}

	@Override
	public void run(CommandLine commandLine) throws Exception {
		Problem problem = null;
		PrintStream output = null;
		ResultFileReader reader = null;

		try {
			// setup the problem
			if (commandLine.hasOption("problem")) {
				problem = ProblemFactory.getInstance().getProblem(commandLine
						.getOptionValue("problem"));
			} else {
				problem = new ProblemStub(Integer.parseInt(commandLine
						.getOptionValue("dimension")));
			}

			try {
				// setup the output stream
				if (commandLine.hasOption("output")) {
					output = new PrintStream(new File(commandLine
							.getOptionValue("output")));
				} else {
					output = System.out;
				}

				// display info for all result files
				for (String filename : commandLine.getArgs()) {
					try {
						int count = 0;
						reader = new ResultFileReader(problem, new File(
								filename));

						while (reader.hasNext()) {
							reader.next();
							count++;
						}

						output.println(filename + " " + count);
					} finally {
						if (reader != null) {
							reader.close();
						}
					}
				}
			} finally {
				if ((output != null) && (output != System.out)) {
					output.close();
				}
			}
		} finally {
			if (problem != null) {
				problem.close();
			}
		}
	}
	
	/**
	 * Starts the command line utility for counting the number of entries in a 
	 * result file.
	 * 
	 * @param args the command line arguments
	 * @throws Exception if an error occurred
	 */
	public static void main(String[] args) throws Exception {
		new ResultFileInfo().start(args);
	}

}
